/* Copyright (C) 1999 Network Associates, Inc.
   All Rights Reserved.

   Author: Michael_Elkins@nai.com
   Last Edit: March 22, 1999 */

/* This file contains code implementing the "Basic Path Validitation"
   algorithm described in section 6.1 of RFC2459 */

#include "cms.h"

#define TC_F_PERMIT         0x010000

/*****
 *
 * tc_find_crl
 *
 * Inputs:
 *	issuer
 *		Certificate for issuer of crl.  this is a pointer to the
 *		certificate, and not just the DN since the cert is required
 *		to verify the CRL.
 *
 *	ctx
 *		CMS context structure.
 *
 * Outputs:
 *	crl
 *		Pointer to requested CRL.  NOTE: this is a pointer to the
 *		in-store copy of the CRL and the application SHOULD NOT free
 *		it.
 *
 * Returns:
 *	0	success
 *	<0	failure
 */

static int
tc_find_crl (TC_CertificateList **crl, /* OUT */
	     TC_CERT *issuer, /* IN */
	     TC_CONTEXT *ctx) /* IN */
{
    int status=TC_E_NOTFOUND; /*crl not found*/
    TC_CertificateList *list;
    TC_LIST *entry;

    if(!crl||!issuer||!ctx)
	return TC_E_INVARGS;

    *crl=NULL;/*clear input*/

    for (entry=ctx->crl; entry; entry=entry->next)
    {
	list = (TC_CertificateList *) entry->data;
	if (tc_compare_dname (&list->tbsCertList.issuer,
			      &issuer->tbsCertificate->issuer,
			      ctx))
	{
	    status=0;
	    *crl=list;
	    break;
	}
    }
    return status;
} /* tc_find_crl */

/*****
 *
 * tc_validate_crl
 *
 * Performs integrity check and PKIX compliance test on specified CRL.
 *
 * Inputs:
 *	crl
 *		CRL to validate.
 *
 *	issuer
 *		Certificate of CRL issuer.
 *
 *	flags
 *		TC_F_PKIX
 *			Turn on pedantic checks required by RFC2459 for CRLs.
 *
 *	ctx
 *		CMS context structure.
 *
 * Returns:
 *	0	success
 *	<0	failure
 */

static int
tc_validate_crl (TC_CertificateList *crl,
		 TC_CERT *issuer,
		 int flags,
		 TC_CONTEXT *ctx)
{
    unsigned char *blob;
    size_t blobsize;
    TC_Extension *ext;
    int status=0, n;

    /* verify signature algorithms are the same on the inside and outside of
       signed block */
    if (tc_compare_block (&crl->signatureAlgorithm.algorithm,
			  &crl->tbsCertList.signature.algorithm) != 1)
	return TC_E_INVALIDCRL; /* signature OIDs do not match */

    if (flags & TC_F_PKIX)
    {
	/* verify version 2 CRL */
	if (!crl->tbsCertList.version ||
	    PKIGetIntVal (ctx->certasnctx, crl->tbsCertList.version, &status) != 1)
	    return TC_E_NOTV2CRL;
	if (status)
	    return (compiler2tc_error (status));

	/* TODO: RFC2459 Section 5.1.2.4 states that for times <2050A.D. CAs
	   MUST use UTCTime, otherwise use GeneralizedTime for >2050A.D.

	   CMS uses `time_t' which on most systems overflows in the year
	   2038. */

	/* verify the existence of the `nextUpdate' field */
	if(!crl->tbsCertList.nextUpdate)
	    return TC_E_MISSINGNEXTUPDATE;

	/* ----- check for mandatory crl extensions ----- */
	status=tc_find_extension(&ext,
				 crl->tbsCertList.crlExtensions,
				 PKIid_ce_authorityKeyIdentifier_OID,
				 PKIid_ce_authorityKeyIdentifier_OID_LEN,
				 ctx);
	if(status)
	    return TC_E_MISSINGAUTHKEYIDEXT;

	status=tc_find_extension(&ext,
				 crl->tbsCertList.crlExtensions,
				 PKIid_ce_cRLNumber_OID,
				 PKIid_ce_cRLNumber_OID_LEN,
				 ctx);
	if(status)
	    return TC_E_MISSINGCRLNUMBEREXT;
    }

    /* verify signature */
    blobsize=PKISizeofTBSCertList(ctx->certasnctx,&crl->tbsCertList,PKITRUE);
    blob=TC_Alloc(ctx->memMgr,blobsize);
    if(!blob)
	return TC_E_NOMEMORY;
    PKIPackTBSCertList(ctx->certasnctx,blob,blobsize,&crl->tbsCertList,&status);
    if(!status)
    {
	status = ctx->verify(blob,
			     blobsize,
			     crl->signatureAlgorithm.algorithm.val,
			     crl->signatureAlgorithm.algorithm.len,
			     crl->signatureAlgorithm.parameters ? crl->signatureAlgorithm.parameters->val : NULL,
			     crl->signatureAlgorithm.parameters ? crl->signatureAlgorithm.parameters->len : 0,
			     crl->signature.val,
			     crl->signature.len,
			     issuer->tbsCertificate->subjectPublicKeyInfo.subjectPublicKey.val,
			     issuer->tbsCertificate->subjectPublicKeyInfo.subjectPublicKey.len,
			     issuer,
			     ctx->verfuncdata,
			     ctx);

    }

    /* process crl extensions */
    if (status == 0 && crl->tbsCertList.crlExtensions)
    {
	status=tc_process_extensions(crl->tbsCertList.crlExtensions,
				     (void*) crl,
				     ctx);
    }

    /* process CRL entry extensions */
    if (status == 0 && crl->tbsCertList.revokedCertificates)
    {
	for (n=0; n < crl->tbsCertList.revokedCertificates->n; n++)
	{
	    status = tc_process_extensions (crl->tbsCertList.revokedCertificates->elt[n]->crlEntryExtensions,
					    (void *) crl,
					    ctx);
	    if (status)
		break;
	}
    }

    TC_Free(ctx->memMgr,blob);
    return status;
} /* tc_validate_crl */

/*****
 *
 * tc_cert_not_revoked
 *
 * Checks CRL for presence of specified certificate.  This function assumes
 * the caller has already checked the CRL using tc_validate_crl().
 *
 * Inputs:
 *	cert
 *		certificate to validate
 *	crl
 *		revocation list to check
 *	when
 *		time at which to check for revocation
 *	flags
 *		should be 0 (none specified)
 *	ctx
 *		CMS context structure
 *
 * Returns:
 *	0	success (certificate is not revoked)
 *	<0	error or certificate is revoked
 */

static int
tc_cert_not_revoked (TC_CERT *cert,
		     TC_CertificateList *crl,
		     time_t when,
		     int flags,
		     TC_CONTEXT *ctx)
{
    int i,status;
    time_t revoked;

    (void) flags;
    /* verify that the certificate specified was issued by the issuer of the
       CRL */
    status = tc_compare_dname (&cert->tbsCertificate->issuer,
			       &crl->tbsCertList.issuer,
			       ctx);
    if (status!=1)
	return TC_E_WRONGCRL;

    /* search the revoked certs list for the named cert */
    if (crl->tbsCertList.revokedCertificates)
    {
	for (i=0; i<crl->tbsCertList.revokedCertificates->n; i++)
	{
	    status = tc_compare_serial (&cert->tbsCertificate->serialNumber,
					&crl->tbsCertList.revokedCertificates->elt[i]->userCertificate);
	    if (status==1)
	    {
		revoked = tc_get_choiceoftime (&crl->tbsCertList.revokedCertificates->elt[i]->revocationDate,
					       ctx);
		return ((when>=revoked) ? TC_E_CERTREVOKED : 0);
	    }
	}
    }

    return 0;
} /* tc_cert_not_revoked */

/*****
 *
 * tc_check_rfc822name_constraint
 *
 * Returns:
 *	1	constraint matches
 *	0	constraint does not match
 */

static int
tc_check_rfc822name_constraint (PKIIA5String *n, PKIIA5String *rfc)
{
    if(n->len < rfc->len)
	return 0;
    return((memcmp(rfc->val,n->val + n->len - rfc->len, rfc->len) == 0));
} /* tc_check_rfc822name_constraint */

/* RFC2459: DNS name restrictions are expressed as foo.bar.com.  Any subdomain
   satisfies the name constraint.  For example, www.foo.bar.com would satisfy
   the constraint but bigfoo.bar.com would not. */
static int tc_check_dnsname_constraint (PKIIA5String *n, PKIIA5String *c)
{
    if(n->len < c->len)
	return 0;
    if (memcmp(n->val+n->len-c->len,c->val,c->len))
	return 0;
    if(n->val==c->val)
	return 1;/*exact match*/
    /* ensure this is a valid subdomain, and doesn't match part of another
       name */
    return ((*(n->val+n->len-c->len-1) == '.'));
} /* tc_check_dnsname_constraint */

static int tc_check_uri_constraint (PKIIA5String *url, PKIIA5String *c)
{
    unsigned char *p, *e;
    size_t l;

    p=url->val;
    while ((size_t)(p - url->val) < url->len - 3 &&
	   !(*p == ':' && *(p+1)=='/' && *(p+2)=='/'))
	p++;
    if(*p!=':')
	return TC_E_INVALIDURI;
    p+=3;/*skip over the :// */

    /* now find the end of the hostname */
    e=p;
    while((size_t)(e - url->val) < url->len && *e != '/' && *e != ':')
	e++;

    l=e-p;/*length of hostname */

    if(l<c->len)
	return 0;

    if(memcmp(p+l-c->len,c->val,c->len))
	return 0;

    if(l==c->len)
	return 1;/*exact match*/

    if(*c->val!='.' && *(p+l-c->len-1)!='.')
	return 0; /* not a subdomain */

    return 1;/*valid subdomain */
} /* tc_check_uri_constraint */

static int
tc_check_ipaddress_constraint (PKIOCTET_STRING *ip, PKIOCTET_STRING *c)
{
    size_t i;

    if(ip->len * 2 != c->len)
	return TC_E_INVALIDIPCONSTRAINT;
    for(i=0;i<ip->len;i++)
    {
	if ((ip->val[i] & c->val[i+ip->len]) != (c->val[i] & c->val[i+ip->len]))
	    return 0;
    }
    return 1;
} /* tc_check_ipaddress_constraint */

static int tc_check_name_constraint (PKIGeneralName *name,
				     PKIGeneralSubtree *constraint,
				     TC_CONTEXT *ctx)
{
    int status=0;

    (void)ctx;

    /*NOTE: otherName, ediPartyName and registeredID are not required
      to be handled for PKIX compliance */

    /* TODO: for rfc822Name we also need to check the DN for the EmailAddress
       attribute if the subjectAltName is missing */
    if(name->CHOICE_field_type != constraint->base.CHOICE_field_type)
	return TC_E_WRONGNAMETYPE;

    switch(constraint->base.CHOICE_field_type)
    {
	case 0xa1: /* rfc822Name */
	    status=tc_check_rfc822name_constraint(name->data,
						  constraint->base.data);
	    break;
	case 0xa2: /* dNSName */
	    status=tc_check_dnsname_constraint(name->data,
					       constraint->base.data);
	    break;
	case 0xa3: /* x400Address */
	    status=TC_E_NAMETYPEUNSUPPORTED;/* not supported */
	    break;
	case 0xa4: /* directoryName */
	    /* TODO */
	    break;
	case 0xa6: /* URI */
	    status=tc_check_uri_constraint(name->data,
					   constraint->base.data);
	    break;
	case 0xa7: /* iPAddress */
	    status=tc_check_ipaddress_constraint(name->data,
						 constraint->base.data);
	    break;
    }
    return status;
} /* tc_check_name_constraint */

/*****
 *
 * tc_check_name_constraint
 *
 * Checks whether the given name matches the specified constraint pattern.
 *
 * Input:
 *	name
 *		name to check
 *	constraint
 *		tree of matching names
 *	flags
 *		TC_F_PERMIT
 *			all matched name types MUST match the constraint
 *	ctx
 *		CMS context structure
 *
 * Returns:
 *	0	name is consistent with constraint
 *	<0	error
 */

static int tc_check_name_constraints (PKIGeneralName *name,
				      PKIGeneralSubtrees *constraint,
				      int flags,
				      TC_CONTEXT *ctx)
{
    int n,status=0;

    for(n=0;n<constraint->n;n++)
    {
	status=tc_check_name_constraint(name,constraint->elt[n],ctx);
	if(status==0)
	{
	    /* name type exists, constraint does not match */
	    if(flags & TC_F_PERMIT)
		return TC_E_CONSTRAINTFAIL;
	}
	else if (status==1)
	{
	    /* name type exists, constraint matches */
	    if(!(flags & TC_F_PERMIT))
		return TC_E_CONSTRAINTFAIL;
	}
    }
    return 0;
} /* tc_check_name_constraints */

static int tc_do_name_constraints (PKIGeneralName *sdn,
				   PKISubjectAltName *san,
				   PKIGeneralSubtrees *tree,
				   int flags,
				   TC_CONTEXT *ctx)
{
    int j, status;

    /* first check subject name */
    status=tc_check_name_constraints(sdn, tree, flags, ctx);
    if(status)
	goto error;

    /* check subjectAltName if present */
    if(san)
    {
	for(j=0;j<san->n;j++)
	{
	    status=tc_check_name_constraints(san->elt[j],
					     tree,
					     flags,
					     ctx);
	    if(status)
		goto error;
	}
    }
error:
    return status;
}

/* reverse's certificate chain such that the CA certificate is at index 0 */
static void tc_reverse_chain (TC_CERT **cert, int count) {
    int i;
    TC_CERT *p;

    for (i = 0; i < count / 2 ; i++) {
	p = cert[count - i - 1];
	cert[count - i - 1] = cert[i];
	cert[i] = p;
    }
}

static int
tc_find_policy (PKIOBJECT_ID *oid, PKICertificatePolicies *pol)
{
    int i;

    for (i = 0; i < pol->n; i++)
    {
	if (tc_compare_block (oid, &pol->elt[i]->policyIdentifier) == 1)
	    return i;
    }
    return TC_E_NOTFOUND;
}

static int
tc_verify_policy (PKICertificatePolicies *req, TC_CERT *cert, TC_CONTEXT *ctx)
{
    int status, i, j;
    PKIExtension *ext;
    PKICertificatePolicies *pol;

    status = tc_find_extension (&ext,
				cert->tbsCertificate->extensions,
				PKIid_ce_certificatePolicies_OID,
				PKIid_ce_certificatePolicies_OID_LEN,
				ctx);
    if (status)
	return status;

    if (!req)
	return 0; /* NULL means "any policy" */

    PKIUnpackCertificatePolicies (ctx->certasnctx, &pol, ext->extnValue.val, ext->extnValue.len, &status);
    if (status)
    {
	status = compiler2tc_error (status);
	return status;
    }

    for ( i = 0 ; i < pol->n ; i++ )
    {
	for ( j = 0 ; j < req->n ; i++ )
	{
	    if ( tc_compare_block ( &pol->elt[i]->policyIdentifier,
				    &req->elt[j]->policyIdentifier ) == 1 )
	    {
		PKIFreeCertificatePolicies (ctx->certasnctx, pol);
		return 0;
	    }
	}
    }
    PKIFreeCertificatePolicies (ctx->certasnctx, pol);
    return TC_E_MISSINGPOLICY;
}

static int
tc_policy_intersect (TC_CERT *cert,
		     PKICertificatePolicies **req,
		     TC_CONTEXT *ctx)
{
    PKIExtension *ext;
    PKICertificatePolicies *i, *pol;
    PKIPolicyInformation *piElement;
    int j, status;

    if (cert->tbsCertificate->extensions == NULL)
	return 0;

    status = tc_find_extension (&ext,
				cert->tbsCertificate->extensions,
				PKIid_ce_certificatePolicies_OID,
				PKIid_ce_certificatePolicies_OID_LEN,
				ctx);
    if (status)
	return ((status == TC_E_NOTFOUND) ? 0 /* no error */ : status);

	    if (ext->critical && (ext->critical->val != 0)) {
		PKIUnpackCertificatePolicies (ctx->certasnctx,
					      &pol,
					      ext->extnValue.val,
					      ext->extnValue.len,
					      &status);
		if (status) {
		    status = compiler2tc_error (status);
		    return status;
		}

		i = PKINewCertificatePolicies (ctx->certasnctx);

		for (j = 0; j < pol->n; j++) {
		    if (tc_find_policy (&pol->elt[j]->policyIdentifier, *req) >= 0) {
			piElement = PKINewPolicyInformation(ctx->certasnctx);
			PKIPutOctVal (ctx->certasnctx,
				      &piElement->policyIdentifier,
				      pol->elt[j]->policyIdentifier.val,
				      pol->elt[j]->policyIdentifier.len);
			PKIAddOfElement(ctx->certasnctx, piElement, i);
			/*i->elt[i->n] = PKINewPolicyInformation (ctx->certasnctx);
			PKIPutOctVal (ctx->certasnctx, &i->elt[i->n]->policyIdentifier,
				      pol->elt[j]->policyIdentifier.val,
				      pol->elt[j]->policyIdentifier.len);
			i->n++;*/
		    }
		}

		PKIFreeCertificatePolicies (ctx->certasnctx, pol);

		if (i->n == 0) {
		    PKIFreeCertificatePolicies (ctx->certasnctx, i);
		    return TC_E_INVALIDPOLICY; /* doesn't match acceptable use */
		}

		PKIFreeCertificatePolicies (ctx->certasnctx, *req);
		*req = i;
	    }

    return 0;
}

static int
tc_process_basic_constraints (TC_CERT *cert, int *maxPathLen, TC_CONTEXT *ctx)
{
    PKIBasicConstraints *bc;
    PKIExtension *ext;
    int n, status;

    status = tc_find_extension (&ext,
				cert->tbsCertificate->extensions,
				PKIid_ce_basicConstraints_OID,
				PKIid_ce_basicConstraints_OID_LEN,
				ctx);
    if (status)
	return 0; /* TODO: should this blindly accept any cert as a CA? */

    PKIUnpackBasicConstraints (ctx->certasnctx,
			       &bc,
			       ext->extnValue.val,
			       ext->extnValue.len,
			       &status);
    if (status)
	return (compiler2tc_error (status));

    if (bc->cA != NULL && bc->cA->val != 0)
    {
	if (bc->pathLenConstraint)
	{
	    n = PKIGetIntVal (ctx->certasnctx, bc->pathLenConstraint, &status);
	    /* `n' is the number of CA certs which can appear, so we can have
	       `n'+1 certs in the chain */
	    if (n + 1 < *maxPathLen)
		*maxPathLen = n + 1;
	}
    }

    PKIFreeBasicConstraints (ctx->certasnctx, bc);
    return 0;
}

static int
tc_copy_GeneralName (PKIGeneralName *dst, PKIGeneralName *src, TC_CONTEXT *ctx)
{
    PKIVariableBlock *asn, *srcasn;

    dst->CHOICE_field_type = src->CHOICE_field_type;
    switch (dst->CHOICE_field_type)
    {
	case 0xa1:
	case 0xa2:
	case 0xa6:
	case 0xa7:
	case 0xa8:
	    srcasn = (PKIVariableBlock *) src->data;
	    asn = PKINewVariableBlock (ctx->certasnctx);
	    PKIPutOctVal (ctx->certasnctx, asn, srcasn->val, srcasn->len);
	    dst->data = (void *) asn;
	    break;
	default:
	    return TC_E_COPYNOTSUPPORTED;
    }
    return 0;
}

/* returns the union of 'a' and 'b' as 'u' */
static int
tc_general_subtrees_union (PKIGeneralSubtrees *u, /* OUT */
			   PKIGeneralSubtrees *a, /* IN */
			   PKIGeneralSubtrees *b, /* IN */
			   TC_CONTEXT *ctx)
{
    int i, j, status = 0;
    PKIGeneralSubtree *element;

    /* first copy 'a' to 'u' */
    for (i = 0; i < a->n ; i++) {
	element = PKINewGeneralSubtree (ctx->certasnctx);
	status = tc_copy_GeneralName (&u->elt[u->n-1]->base,
				      &element->base,
				      ctx);
	if (status) {
	    PKIFreeGeneralSubtree(ctx->certasnctx, element);
	    goto error;
	}
	PKIAddOfElement(ctx->certasnctx, element, u);
	/*u->elt[u->n] = PKINewGeneralSubtree (ctx->certasnctx);
	u->n++;
	status = tc_copy_GeneralName (&u->elt[u->n-1]->base,
				      &a->elt[i]->base,
				      ctx);
	if (status)
	    goto error;*/
    }

    /* add missing elements of 'b' to 'u' */
    for (i = 0; i < b->n; i++) {
	for (j = 0; j < u->n; j++) {
	    if (tc_compare_GeneralName (&b->elt[i]->base, &u->elt[j]->base, ctx) == 1)
		break;
	}
	if (j == u->n) {
	    element = PKINewGeneralSubtree (ctx->certasnctx);
	    status = tc_copy_GeneralName (&u->elt[u->n-1]->base,
				      &element->base,
				      ctx);
	    if (status) {
		PKIFreeGeneralSubtree(ctx->certasnctx, element);
		goto error;
	    }
	    PKIAddOfElement(ctx->certasnctx, element, u);
	    /*u->elt[u->n] = PKINewGeneralSubtree (ctx->certasnctx);
	    u->n++;
	    status = tc_copy_GeneralName (&u->elt[u->n-1]->base, &b->elt[i]->base,ctx);
	    if (status)
		goto error;*/
	}
    }

error:
    if (status) {
	for (i = 0; i < u->n; i++)
	    PKIFreeGeneralSubtree (ctx->certasnctx, u->elt[i]);
    }
    return status;
}

static int
tc_name_constraints_intersect (TC_CERT *cert,
			       PKIGeneralSubtrees **constraints,
			       PKIGeneralSubtrees **exclude,
			       TC_CONTEXT *ctx)
{
    PKINameConstraints *nc;
    PKIExtension *ext;
    PKIGeneralSubtrees *gs=NULL;
    PKIGeneralSubtree *element;
    int i, j, status;

    if (cert->tbsCertificate->extensions == NULL)
	return 0;

    status = tc_find_extension (&ext,
				cert->tbsCertificate->extensions,
				PKIid_ce_nameConstraints_OID,
				PKIid_ce_nameConstraints_OID_LEN,
				ctx);
    if (status)
	return ((status == TC_E_NOTFOUND) ? 0 : status);

    PKIUnpackNameConstraints (ctx->certasnctx,
			      &nc,
			      ext->extnValue.val,
			      ext->extnValue.len,
			      &status);
    if (status)
	return (compiler2tc_error (status));

    if (nc->permittedSubtrees) {
	if (*constraints == NULL) {
	    *constraints = nc->permittedSubtrees;
	    nc->permittedSubtrees = NULL;
	}
	else {
	    /* find the intersection */

	    gs = PKINewGeneralSubtrees (ctx->certasnctx);

	    for (i = 0; i < nc->permittedSubtrees->n; i++) {
		for (j = 0; j < (*constraints)->n; j++) {
		    /* TODO: if the compare doesn't understand one of the
		       general name types (like EDIPartyName), it will
		       silently be ignored, what to do in that case? */
		    if (tc_compare_GeneralName (&nc->permittedSubtrees->elt[i]->base,
						&(*constraints)->elt[j]->base,
						ctx) == 1)
		    {
			element = PKINewGeneralSubtree (ctx->certasnctx);
			status = tc_copy_GeneralName (&element->base,
						      &nc->permittedSubtrees->elt[i]->base,
						      ctx);
			if (status) {
			    PKIFreeGeneralSubtree (ctx->certasnctx, element);
			    goto error;
			}
			PKIAddOfElement(ctx->certasnctx, element, gs);
			/*gs->elt[gs->n] = PKINewGeneralSubtree (ctx->certasnctx);
			gs->n++;
			status = tc_copy_GeneralName (&gs->elt[gs->n-1]->base,
						      &nc->permittedSubtrees->elt[i]->base,
						      ctx);
			if (status)
			    goto error;*/
		    }
		}
	    }

	    PKIFreeGeneralSubtrees (ctx->certasnctx, *constraints);
	    *constraints = gs;
	    gs = NULL;
	}
    }

    if (nc->excludedSubtrees) {
	if (*exclude == NULL) {
	    *exclude = nc->excludedSubtrees;
	    nc->excludedSubtrees = NULL;
	}
	else {
	    /* find the union of the two trees */
	    status = tc_general_subtrees_union (gs,
						nc->excludedSubtrees,
						*exclude,
						ctx);
	    PKIFreeGeneralSubtrees (ctx->certasnctx, *exclude);
	    *exclude = gs;
	    gs = NULL;
	}
    }

error:
    if (gs)
	PKIFreeGeneralSubtrees (ctx->certasnctx, gs);
    if (nc)
	PKIFreeNameConstraints (ctx->certasnctx, nc);
    return status;
}

static int
tc_process_policy_constraints (TC_CERT *cert,
			       int i,
			       int *explicit,
			       int *mapping,
			       TC_CONTEXT *ctx)
{
    PKIPolicyConstraints *pc;
    PKIExtension *ext;
    int status;

    status = tc_find_extension (&ext,
				cert->tbsCertificate->extensions,
				PKIid_ce_policyConstraints_OID,
				PKIid_ce_policyConstraints_OID_LEN,
				ctx);
    if (status)
	return 0;

    PKIUnpackPolicyConstraints (ctx->certasnctx,
				&pc,
				ext->extnValue.val,
				ext->extnValue.len,
				&status);
    if (status)
	return (compiler2tc_error (status));

    if (pc->requireExplicitPolicy)
    {
	int r = PKIGetIntVal (ctx->certasnctx, pc->requireExplicitPolicy, &status);

	if (status)
	{
	    status = compiler2tc_error (status);
	    goto error;
	}

	if (i + r < *explicit)
	    *explicit = i + r;
    }

    if (pc->inhibitPolicyMapping)
    {
	int q = PKIGetIntVal (ctx->certasnctx, pc->inhibitPolicyMapping, &status);

	if (status)
	{
	    status = compiler2tc_error (status);
	    goto error;
	}
	if (i + q < *mapping)
	    *mapping = i + q;
    }

error:

    PKIFreePolicyConstraints (ctx->certasnctx, pc);
    return status;
}

static int
tc_process_key_usage (TC_CERT *cert, TC_CONTEXT *ctx)
{
    PKIKeyUsage *ku;
    PKIExtension *ext;
    int bits=0, status;
    size_t i;

    status = tc_find_extension (&ext,
				cert->tbsCertificate->extensions,
				PKIid_ce_keyUsage_OID,
				PKIid_ce_keyUsage_OID_LEN,
				ctx);
    if (status)
	return 0;

    PKIUnpackKeyUsage (ctx->certasnctx,
		       &ku,
		       ext->extnValue.val,
		       ext->extnValue.len,
		       &status);
    if (status)
	return (compiler2tc_error (status));

    for (i=0;i<ku->len;i++)
    {
	bits <<= 8;
	bits |= ku->val[i];
    }
    bits >>= ku->nuub;

    status = (bits & PKIKeyUsage_keyCertSign) ? 0 : TC_E_NOTSIGNINGKEY;

    PKIFreeKeyUsage (ctx->certasnctx, ku);

    return status;
}

/*****
 *
 * tc_validate_path
 *
 * Inputs:
 *	cert
 *		end entity certificate of path to validate
 *
 *	when
 *		time at which validity is being checked for.  According to
 *		RFC2459, this can be "the current date/time, or some point in
 *		the past."  Future validity can not be determined since the
 *		certificate could be revoked at any time.
 *
 *	policy
 *		acceptable certificate policies for this validation (not
 *		implemented)
 *
 *  flags
 *		TC_F_PKIX
 *			enables stricter PKIX certificate checking
 *
 *	ctx
 *		CMS context structure, which should contain the necessary
 *		certificates and CRLs required for validation of the path.
 *
 * Returns:
 *	0	success
 *	<0	failure
 */

int tc_validate_path (TC_CERT *eecert,
		      PKICertificatePolicies *initialPolicies,
		      time_t when,
		      int flags,
		      TC_CONTEXT *ctx)
{
    /* STATE VARIABLES */

    PKICertificatePolicies *acceptablePolicies=NULL;
    PKIGeneralSubtrees *constrainedSubtrees=NULL;
    PKIGeneralSubtrees *excludedSubtrees=NULL;
    int explicitPolicy;
    int policyMapping;
    int maxPathLen; /*basicConstraints*/

    /* local variables */

    TC_CERT *cert;
    int i,j,status=0, chainsize=0;
    time_t notBefore,notAfter;
    TC_CertificateList *crl;
    TC_CERT **chain=NULL;
    PKIGeneralName dummy;
    PKISubjectAltName *san=NULL;

    /* PKIX path validation begins with the CA certificate, so we must first
       locate it in the store. */

    TC_Realloc (ctx->memMgr,(void**)&chain,sizeof(TC_CERT *)*(chainsize+1));
    chain[chainsize++] = eecert;

    while (!tc_is_self_signed (chain[chainsize-1], ctx))
    {
	status = tc_find_issuer(&cert,chain[chainsize-1],ctx);
	if (status)
	    goto error;
	TC_Realloc (ctx->memMgr,(void**)&chain,sizeof(TC_CERT *)*(chainsize+1));
	chain[chainsize++]=cert;
    }

    explicitPolicy = chainsize;
    policyMapping = chainsize;
    maxPathLen = chainsize;

    /* we need to process the chain from the root (CA) certificate down to the
       end entity certificate in question.  this is necessary since the CA
       certificate(s) may have certain restrictions which need to be processed
       before end entity validity can be determined. */
    tc_reverse_chain (chain, chainsize);

    /* loop through all certificate in the chain, starting with the CA cert */
    for (i = 0; i < chainsize; i++)
    {
	/* if the maximum number of certificates in the change has been
	   exceeded, the certificate is not valid */
	if (maxPathLen == 0)
	{
	    status = TC_E_PATHCONSTRAINT;
	    goto error;
	}
	maxPathLen--;

	/* ----- verify basic certificate information ----- */

	/* check version number and format of message */
	if (!chain[i]->tbsCertificate->version)
	{
	    /* version 1 */
	    if (chain[i]->tbsCertificate->subjectUniqueID ||
	        chain[i]->tbsCertificate->issuerUniqueID ||
		chain[i]->tbsCertificate->extensions)
	    {
		/* version 1 x.509 certs should not have these fields */
		status = TC_E_NOTV1CERT;
		goto error;
	    }
	}
	else
	{
	    int version = PKIGetIntVal(ctx->certasnctx,
				       chain[i]->tbsCertificate->version,
				       &status);

	    if (version == 0)
	    {
		/* if we have a version 1 cert, the version field should be
		   absent */
		status = TC_E_EXPLICITV1;
		goto error;
	    }
	    else if (version == 1) /* v2 */
	    {
		if ((!chain[i]->tbsCertificate->subjectUniqueID &&
		    !chain[i]->tbsCertificate->issuerUniqueID) ||
		    chain[i]->tbsCertificate->extensions)
		{
		    status = TC_E_NOTV2CERT;
		    goto error;
		}
	    }
	    else if (version == 2) /* v3 */
	    {
		/* v3 certs should always have at least one extension */
		if (!chain[i]->tbsCertificate->extensions)
		{
		    status = TC_E_NOTV3CERT;
		    goto error;
		}
	    }
	}

	/* verify that the inner and outer signature algorithms OIDs are
	   the same */
	if (!tc_compare_block (&chain[i]->tbsCertificate->signature.algorithm,
			       &chain[i]->cert->signatureAlgorithm.algorithm))
	{
	    /* OIDs differ */
	    status = TC_E_SIGALGMISMATCH;
	    goto error;
	}

	/*if we are processing a self-signed cert, it is its own issuer*/
	cert = (i == 0) ? chain[0] : chain[i-1];

	/* (a) (1) verify the signature on the subject certificate*/
	status = ctx->verify(chain[i]->cert->tbsCertificate.val,
			     chain[i]->cert->tbsCertificate.len,
			     chain[i]->cert->signatureAlgorithm.algorithm.val,
			     chain[i]->cert->signatureAlgorithm.algorithm.len,
			     chain[i]->cert->signatureAlgorithm.parameters ? chain[i]->cert->signatureAlgorithm.parameters->val : NULL,
			     chain[i]->cert->signatureAlgorithm.parameters ? chain[i]->cert->signatureAlgorithm.parameters->len : 0,
			     chain[i]->cert->signature.val,
			     chain[i]->cert->signature.len,
			     cert->tbsCertificate->subjectPublicKeyInfo.subjectPublicKey.val,
			     cert->tbsCertificate->subjectPublicKeyInfo.subjectPublicKey.len,
			     cert,
			     ctx->verfuncdata,
			     ctx);

	if (status)
	    goto error;

	/* (a) (2) verify the validity period includes `when'*/
	notBefore=tc_get_choiceoftime(&chain[i]->tbsCertificate->validity.notBefore,ctx);
	notAfter=tc_get_choiceoftime(&chain[i]->tbsCertificate->validity.notAfter,ctx);
	if (when > time(0) || when < notBefore || when > notAfter)
	{
	    /* outside of validity period */
	    status=TC_E_INVDATE;
	    goto error;
	}

	/* locate the signer's crl */
	status = tc_find_crl (&crl, cert, ctx);
	if (status == 0)
	{
	    /*validate the CRL*/
	    status = tc_validate_crl (crl,cert,flags,ctx);
	    if (status)
		goto error;

	    /* (a) (3) verify certificate was not revoked at time `when'*/
	    status = tc_cert_not_revoked (chain[i],crl,when,0,ctx);
	    if (status)
		goto error;
	}
	else if (status != TC_E_NOTFOUND)
	    goto error;

	/* (a) (4) verify subject and issuer names chain correctly.

	   this is implicitly done above in bulding the the array of
	   certificates to validate. */

	/* locate the subjectAltName extension for following checks */
	if (cert->tbsCertificate->extensions)
	{
	    PKIExtension *ext;

	    status=tc_find_extension(&ext,
				     cert->tbsCertificate->extensions,
				     PKIid_ce_subjectAltName_OID,
				     PKIid_ce_subjectAltName_OID_LEN,
				     ctx);
	    if (status == 0)
	    {
		PKIUnpackSubjectAltName(ctx->certasnctx,
					&san,
					ext->extnValue.val,
					ext->extnValue.len,
					&status);
		if (status)
		{
		    status=compiler2tc_error(status);
		    goto error;
		}
	    }
	}
	else if (flags & TC_F_PKIX)
	{
	    /* if we are doing pedantic PKIX checking, the fact that there are
	       no certificates extensions is an error since all certificates
	       are required to have at least the AuthorityKeyIdentifier
	       extension */
	    goto error;
	}
	else
	    san = NULL; /* subjectAltName */

	/* create dummy general name with the DN in it */
	memset (&dummy, 0, sizeof (dummy));
	dummy.CHOICE_field_type = 0xa4; /* directoryName */
	dummy.data = (void*)&chain[i]->tbsCertificate->subject;

	/* (b) verify that the subject name and subjectAltName extension
	   (critical or not) is consistent with the constrained subtrees
	   state variable. */
	if (constrainedSubtrees)
	{
	    status=tc_do_name_constraints(&dummy,
					  san,
					  constrainedSubtrees,
					  TC_F_PERMIT,
					  ctx);
	    if(status)
		goto error;
	}

	/* (c) verify that the subject name and subjectAltName extension
	   (critical or not) is consistent with the excluded subtrees
	   state variables */

	if (excludedSubtrees) {
	    status = tc_do_name_constraints (&dummy,
					     san,
					     excludedSubtrees,
					     0,
					     ctx);
	    if (status)
		goto error;
	}

	/* (d) verify that policy information is consistent with the initial
	   policy set. */

	/* (d) (1) */
	if (explicitPolicy <= i) {
	    status = tc_verify_policy (initialPolicies, chain[i], ctx);
	    if (status)
		goto error;
	}

	/* (d) (2) ensure policy is not mapped */
	if (policyMapping <= i) {
	    PKIExtension *ext;

	    status = tc_find_extension (&ext,
					chain[i]->tbsCertificate->extensions,
					PKIid_ce_policyMappings_OID,
					PKIid_ce_policyMappings_OID_LEN,
					ctx);
	    if (status == 0) {
		status = TC_E_POLICYMAPPED;
		goto error;
	    }
	}

	/* (e) verify that the policy information is consistent with the
	   acceptable policy set: */

	/* (e) (1) if certificate policies extension is marked critical,
	   the intersection of the policies extension and the acceptable
	   policy set shall be non-null */

	/* (e) (2) the acceptable policy set is assigned the resulting
	   intersection as its new value. */

	status = tc_policy_intersect (chain[i], &acceptablePolicies, ctx);
	if (status)
	    goto error;

	/* (g) verify that the intersection of the acceptable policy set
	   and initial policy set is non-null */

	if (initialPolicies && acceptablePolicies) {
	    status = TC_E_POLICYMISMATCH;
	    for ( j = 0 ; j < initialPolicies->n ; j++ ) {
		if (tc_find_policy (&initialPolicies->elt[j]->policyIdentifier,
				    acceptablePolicies) >= 0) {
		    status = 0;
		    break;
		}
	    }
	    if (status)
		goto error;
	}

	/* (h) recognize and process any other critical extensions present
	   in the certificate */

	status = tc_process_extensions (chain[i]->tbsCertificate->extensions,
					(void *) chain[i],
					ctx);
	if (status)
	    goto error;

	/* the following checks only apply to CA certificates */
	if (i < chainsize - 1) {
	    /* (i) verify that the certificate is a CA certificate
	       (as specified in a basicConstraints extension or as verified
	       out-of-band). */

	    status = tc_process_basic_constraints (chain[i], &maxPathLen, ctx);
	    if (status)
		goto error;

	    /* (j) if permittedSubtrees is present in the certificate, set
	       the constrained subtrees state variable to the intersection
	       of its previous value and the value indicated in the
	       extension field */

	    /* (k) if excludedSubtrees is present in the certificate, set
	       the excluded subtrees variable stat variable to the union of its
	       previous value and the value indicated in the extension field. */

	    status = tc_name_constraints_intersect (cert,
						    &constrainedSubtrees,
						    &excludedSubtrees,
						    ctx);
	    if (status)
		goto error;

	    /* (l) if a policy constraints extension is included in the
	       certificate, modify the explicit policy and policy mapping
	       state variables as follows:

	       (l) (1) if requireExplicitPolicy is present and has value r,
	       the explicit policy mapping state variable is set to the
	       minimum of its current value and the sum of r and i (the
	       current certificate in the sequence).

	       (l) (2) if inhibitPolicyMapping is present and has value q,
	       the policy mapping state variable is set to the minimum of
	       its current value and the sum of q and i (the current
	       certificate in the sequence). */

	    status = tc_process_policy_constraints (cert,
						    i,
						    &explicitPolicy,
						    &policyMapping,
						    ctx);
	    if (status)
		goto error;

	    /* (m) if a key usage extension is marked critical, ensure the
	       keyCertSign bit is set */

	    status = tc_process_key_usage (cert, ctx);
	    if (status)
		goto error;
	}

	/* clean up */
	PKIFreeSubjectAltName(ctx->certasnctx,san);
	san=NULL;
    }

error:
    if(chain)
	TC_Free(ctx->memMgr,chain);
    if(san)
	PKIFreeSubjectAltName(ctx->certasnctx,san);
    return status;
} /* tc_validate_path */
